/**
 * Template Saver Service
 * บริการสำหรับบันทึกเทมเพลต
 */
(function(global) {
  'use strict';

  /**
   * คลาส TemplateSaver
   * บริการสำหรับบันทึกเทมเพลต
   */
  class TemplateSaver {
    constructor(editor) {
      this.editor = editor;
      this.autoSaveTimer = null;
      this.lastSaveTime = null;
    }

    /**
     * บันทึกเทมเพลต
     */
    save() {
      const html = this.editor.getCleanHTML();
      const css = this.getCSS();
      const metadata = this.getMetadata();

      // สร้างออบเจกต์เทมเพลต
      const template = {
        html,
        css,
        metadata,
        timestamp: new Date().toISOString()
      };

      // บันทึกใน localStorage
      if (this.editor.services.storageService) {
        this.editor.services.storageService.save('template', template);
      } else {
        // การใช้งานสำรอง
        localStorage.setItem('editor-template', JSON.stringify(template));
      }

      // อัปเดตเวลาบันทึกล่าสุด
      this.lastSaveTime = new Date();

      // ส่งเหตุการณ์
      this.editor.emit('template:saved', {template});

      // แสดงการแจ้งเตือน
      this.showSaveNotification();
    }

    /**
     * บันทึกอัตโนมัติ
     */
    autoSave() {
      // ล้างตัวจับเวลาที่มีอยู่
      if (this.autoSaveTimer) {
        clearTimeout(this.autoSaveTimer);
      }

      // ตั้งตัวจับเวลาใหม่
      this.autoSaveTimer = setTimeout(() => {
        this.save();
      }, 5000); // บันทึกอัตโนมัติหลังจาก 5 วินาที
    }

    /**
     * รับ CSS ทั้งหมด
     */
    getCSS() {
      let css = '';

      // รับ CSS จากปลั๊กอินทั้งหมด
      Object.keys(this.editor.plugins).forEach(pluginName => {
        const plugin = this.editor.plugins[pluginName];
        if (typeof plugin.getCSS === 'function') {
          css += `/* ${pluginName} Plugin CSS */\n`;
          css += plugin.getCSS();
          css += '\n\n';
        }
      });

      // รับ CSS จากสไตล์ชีตที่กำหนดเอง
      const customStyles = document.querySelectorAll('style[data-editor-custom]');
      customStyles.forEach(style => {
        css += `/* Custom CSS */\n`;
        css += style.textContent;
        css += '\n\n';
      });

      return css;
    }

    /**
     * รับข้อมูลเมตาดาต้า
     */
    getMetadata() {
      return {
        title: document.title || 'Untitled Template',
        author: 'Website Template Editor',
        version: '1.0.0',
        description: 'Template created with Website Template Editor',
        tags: ['template', 'editor'],
        components: Object.keys(this.editor.plugins)
      };
    }

    /**
     * แสดงการแจ้งเตือนการบันทึก
     */
    showSaveNotification() {
      // สร้างการแจ้งเตือน
      const notification = this.editor.domUtils.createElement('div', {
        'class': 'editor-save-notification',
        'id': 'editor-save-notification'
      }, 'เทมเพลตถูกบันทึกแล้ว');

      // เพิ่มในเอกสาร
      document.body.appendChild(notification);

      // แสดงการแจ้งเตือน
      setTimeout(() => {
        notification.classList.add('editor-notification-show');
      }, 10);

      // ซ่อนการแจ้งเตือนหลังจาก 3 วินาที
      setTimeout(() => {
        notification.classList.remove('editor-notification-show');
        setTimeout(() => {
          document.body.removeChild(notification);
        }, 300);
      }, 3000);
    }
  }

  // เปิดเผยคลาส TemplateSaver ทั่วโลก
  global.TemplateSaver = TemplateSaver;

})(typeof window !== 'undefined' ? window : this);