/**
 * script.js - ไฟล์ JavaScript สำหรับเชื่อมต่อกับ PHP API
 */

// สร้างอินสแตนซ์ของ TreeManager
let treeManager;

// เก็บไว้ตรวจสอบความเปลี่ยนแปลง
let lastSavedData = '';

// ฟังก์ชันเริ่มต้นการทำงาน
document.addEventListener('DOMContentLoaded', () => {
  // สร้าง TreeManager
  initTreeManager();

  // เพิ่ม event listeners สำหรับปุ่มต่างๆ
  setupEventListeners();

  // โหลดข้อมูลจาก API
  loadTreeDataFromAPI();
});

/**
 * สร้าง TreeManager
 */
function initTreeManager() {
  // กำหนดค่าเริ่มต้น
  const config = {
    containerId: 'tree-container',
    limitedDepth: true,
    maxDepth: 3,
    levelNames: {
      1: 'Stage',
      2: 'Area',
      3: 'Zone'
    },
    outputId: 'json-output'
  };

  // สร้าง TreeManager instance
  treeManager = new TreeManager(config);
}

/**
 * เพิ่ม event listeners
 */
function setupEventListeners() {
  // ปุ่มเพิ่มรายการระดับบนสุด
  document.getElementById('add-root-btn').addEventListener('click', () => {
    treeManager.addRootNode('รายการใหม่');
    saveTreeData();
  });

  // ปุ่มยุบทั้งหมด
  document.getElementById('collapse-all-btn').addEventListener('click', () => {
    treeManager.collapseAll();
  });

  // ปุ่มขยายทั้งหมด
  document.getElementById('expand-all-btn').addEventListener('click', () => {
    treeManager.expandAll();
  });

  // ปุ่มบันทึก
  document.getElementById('save-btn').addEventListener('click', () => {
    saveTreeData();
  });

  // ปุ่มรีเฟรช
  document.getElementById('refresh-btn').addEventListener('click', () => {
    loadTreeDataFromAPI();
  });

  // ตั้งเวลาบันทึกอัตโนมัติทุก 1 นาที
  setInterval(saveTreeDataIfChanged, 60000);
}

/**
 * บันทึกข้อมูลถ้ามีการเปลี่ยนแปลง
 */
function saveTreeDataIfChanged() {
  const currentData = JSON.stringify(treeManager.getTreeData());
  if (currentData !== lastSavedData) {
    saveTreeData();
  }
}

/**
 * โหลดข้อมูลจาก API
 */
async function loadTreeDataFromAPI() {
  try {
    // แสดงข้อความโหลด
    showMessage('กำลังโหลดข้อมูล...', 'info');

    // เรียกใช้ API
    const response = await fetch('api.php?action=export_tree');
    const result = await response.json();

    if (result.status === 'success') {
      // นำข้อมูลมาใช้กับ TreeManager
      treeManager.setTreeData(result.data);

      // เก็บข้อมูลเพื่อเปรียบเทียบการเปลี่ยนแปลง
      lastSavedData = JSON.stringify(treeManager.getTreeData());

      showMessage('โหลดข้อมูลสำเร็จ', 'success');
    } else {
      // ถ้าไม่มีข้อมูล หรือเกิดข้อผิดพลาด
      showMessage('ไม่พบข้อมูล หรือเกิดข้อผิดพลาด - สร้างโครงสร้างเริ่มต้น', 'warning');
      treeManager.generateInitialStructure();
    }
  } catch (error) {
    console.error('เกิดข้อผิดพลาดในการโหลดข้อมูล:', error);
    showMessage('เกิดข้อผิดพลาดในการโหลดข้อมูล - สร้างโครงสร้างเริ่มต้น', 'error');
    treeManager.generateInitialStructure();
  }
}

/**
 * บันทึกข้อมูลไปยัง API
 */
async function saveTreeData() {
  try {
    // แสดงข้อความกำลังบันทึก
    showMessage('กำลังบันทึกข้อมูล...', 'info');

    // ดึงข้อมูลจาก TreeManager
    const treeData = treeManager.getTreeData();

    // เรียกใช้ API
    const response = await fetch('api.php?action=import_tree', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(treeData)
    });

    const result = await response.json();

    if (result.status === 'success') {
      // บันทึกข้อมูลที่บันทึกล่าสุด
      lastSavedData = JSON.stringify(treeData);
      showMessage('บันทึกข้อมูลสำเร็จ', 'success');
    } else {
      showMessage('เกิดข้อผิดพลาดในการบันทึกข้อมูล: ' + (result.message || 'ไม่ทราบสาเหตุ'), 'error');
    }
  } catch (error) {
    console.error('เกิดข้อผิดพลาดในการบันทึกข้อมูล:', error);
    showMessage('เกิดข้อผิดพลาดในการบันทึกข้อมูล: ' + error.message, 'error');
  }
}

/**
 * แสดงข้อความแจ้งเตือน
 * @param {string} message ข้อความ
 * @param {string} type ประเภท ('success', 'info', 'warning', 'error')
 */
function showMessage(message, type) {
  const messageContainer = document.getElementById('message-container');
  if (!messageContainer) return;

  // สร้าง element สำหรับข้อความ
  const messageElement = document.createElement('div');
  messageElement.className = `alert alert-${type}`;
  messageElement.textContent = message;

  // ล้างข้อความเก่า
  messageContainer.innerHTML = '';

  // เพิ่มข้อความใหม่
  messageContainer.appendChild(messageElement);

  // ซ่อนข้อความหลังจาก 3 วินาที ยกเว้นกรณี error
  if (type !== 'error') {
    setTimeout(() => {
      messageElement.style.opacity = '0';
      setTimeout(() => {
        messageContainer.innerHTML = '';
      }, 500);
    }, 3000);
  }
}

/**
 * สร้าง External ID
 * @returns {string} External ID
 */
function generateExternalId() {
  return 'ext_' + Date.now() + '_' + Math.floor(Math.random() * 1000);
}

/**
 * ฟังก์ชันสำหรับนำ ID ของโหนดไปใช้กับระบบอื่น
 * @param {number} nodeId ID ของโหนด
 */
async function useNodeWithExternalSystem(nodeId) {
  try {
    // โหลดข้อมูลโหนด
    const response = await fetch(`api.php?action=get_node&id=${nodeId}`);
    const result = await response.json();

    if (result.status === 'success') {
      const node = result.data;

      // ตรวจสอบว่าเป็นโหนดระดับที่ 3 (Zone) หรือไม่
      if (node.level === 3) {
        // ตรวจสอบว่ามี external_id หรือไม่
        if (!node.external_id) {
          // สร้าง external_id ถ้าไม่มี
          const externalId = generateExternalId();

          // อัพเดตโหนด
          await updateNodeExternalId(nodeId, externalId);

          // อัพเดตข้อมูล node
          node.external_id = externalId;
        }

        // ทำงานกับ external_id
        console.log(`นำ Zone ID: ${node.id} (External ID: ${node.external_id}) ไปใช้กับระบบอื่น`);

        // ตัวอย่างการใช้งาน - แสดงรายละเอียด
        showZoneDetails(node);
      } else {
        showMessage('โหนดที่เลือกไม่ใช่ Zone (ระดับ 3)', 'warning');
      }
    } else {
      showMessage('ไม่พบข้อมูลโหนด', 'error');
    }
  } catch (error) {
    console.error('เกิดข้อผิดพลาด:', error);
    showMessage('เกิดข้อผิดพลาดในการเรียกใช้งาน: ' + error.message, 'error');
  }
}

/**
 * อัพเดต External ID ของโหนด
 * @param {number} nodeId ID ของโหนด
 * @param {string} externalId External ID ใหม่
 */
async function updateNodeExternalId(nodeId, externalId) {
  try {
    // ข้อมูลที่ต้องการอัพเดต
    const updateData = {
      id: nodeId,
      external_id: externalId
    };

    // เรียกใช้ API
    const response = await fetch('api.php?action=update_node', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(updateData)
    });

    const result = await response.json();

    if (result.status !== 'success') {
      console.error('ไม่สามารถอัพเดต External ID:', result.message);
    }
  } catch (error) {
    console.error('เกิดข้อผิดพลาดในการอัพเดต External ID:', error);
  }
}

/**
 * แสดงรายละเอียดของ Zone
 * @param {Object} zone ข้อมูลของ Zone
 */
function showZoneDetails(zone) {
  // รับ element สำหรับแสดงรายละเอียด
  const detailsContainer = document.getElementById('zone-details');
  if (!detailsContainer) return;

  // สร้าง HTML สำหรับแสดงรายละเอียด
  detailsContainer.innerHTML = `
        <div class="zone-detail-card">
            <h3>รายละเอียด Zone</h3>
            <table class="zone-detail-table">
                <tr>
                    <th>Database ID:</th>
                    <td>${zone.id}</td>
                </tr>
                <tr>
                    <th>ชื่อ:</th>
                    <td>${zone.name}</td>
                </tr>
                <tr>
                    <th>External ID:</th>
                    <td>${zone.external_id || '(ไม่มี)'}</td>
                </tr>
                <tr>
                    <th>ระดับ:</th>
                    <td>${zone.level} (Zone)</td>
                </tr>
            </table>
            <div class="zone-detail-actions">
                <button class="btn" onclick="linkToExternalSystem('${zone.id}', '${zone.external_id}')">
                    เชื่อมต่อกับระบบอื่น
                </button>
            </div>
        </div>
    `;

  // แสดง container
  detailsContainer.style.display = 'block';
}

/**
 * เชื่อมต่อกับระบบอื่น (ตัวอย่าง)
 * @param {string} zoneId ID ของ Zone
 * @param {string} externalId External ID ของ Zone
 */
function linkToExternalSystem(zoneId, externalId) {
  // ตัวอย่างการสร้าง URL สำหรับระบบอื่น
  const url = `external-system.php?zone_id=${zoneId}&external_id=${externalId}`;

  // แสดงข้อความ
  showMessage(`กำลังเชื่อมต่อกับระบบอื่น: Zone ID ${zoneId}, External ID ${externalId}`, 'info');

  // เปิด URL ในแท็บใหม่ (ตัวอย่าง)
  // window.open(url, '_blank');

  console.log(`เชื่อมต่อกับ URL: ${url}`);
}