<?php
/**
 * Admin Panel Entry Point
 * Separate entry point for admin routes with enhanced security
 */

// Set error reporting based on environment
if (isset($_ENV['APP_ENV']) && $_ENV['APP_ENV'] === 'production') {
    error_reporting(0);
    ini_set('display_errors', 0);
} else {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
}

// Set timezone
date_default_timezone_set($_ENV['TIMEZONE'] ?? 'Asia/Bangkok');

// Load environment variables if .env file exists
$envFile = __DIR__.'/../.env';
if (file_exists($envFile)) {
    $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
            list($key, $value) = explode('=', $line, 2);
            $_ENV[trim($key)] = trim($value, '"\'');
        }
    }
}

// Include autoloader for API requests
require_once __DIR__.'/../src/Core/Autoloader.php';

// Register autoloader
$autoloader = new \App\Core\Autoloader();
$autoloader->addNamespace('App', __DIR__.'/../src');
$autoloader->register();

// Load configuration
$config = require __DIR__.'/../config/app.php';

// Enhanced security for admin panel
$securityHeaders = [
    'X-Frame-Options: DENY',
    'X-Content-Type-Options: nosniff',
    'X-XSS-Protection: 1; mode=block',
    'Strict-Transport-Security: max-age=31536000; includeSubDomains',
    'Referrer-Policy: strict-origin-when-cross-origin',
    'Content-Security-Policy: default-src \'self\'; script-src \'self\' \'unsafe-inline\'; style-src \'self\' \'unsafe-inline\'; img-src \'self\' data: https:; font-src \'self\' https:'
];

foreach ($securityHeaders as $header) {
    header($header);
}

// IP whitelist check for admin (if configured)
if (!empty($_ENV['ADMIN_IP_WHITELIST'])) {
    $allowedIPs = explode(',', $_ENV['ADMIN_IP_WHITELIST']);
    $clientIP = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'];

    if (!in_array($clientIP, $allowedIPs)) {
        http_response_code(403);
        header('Content-Type: application/json');
        echo json_encode(['error' => 'Access denied']);
        exit;
    }
}

// Set up simple error handling
ini_set('log_errors', 1);
ini_set('error_log', __DIR__.'/../storage/logs/admin-errors.log');

// Initialize router with admin base path
$router = new \App\Core\Router($config['admin_base_path']);

// Add enhanced middleware for admin
$router->addGlobalMiddleware(\App\Middleware\SecurityMiddleware::class);
$router->addGlobalMiddleware(\App\Middleware\CorsMiddleware::class);
$router->addGlobalMiddleware(\App\Middleware\AdminSecurityMiddleware::class);

// Load routes (admin routes will be filtered by middleware)
$routeLoader = require __DIR__.'/../config/routes.php';
$routeLoader($router);

// Handle the request
try {
    $router->dispatch();
} catch (\Exception $e) {
    // Enhanced error handling for admin
    http_response_code(500);
    header('Content-Type: application/json');

    // Log admin errors with more detail
    $errorDetails = [
        'timestamp' => date('Y-m-d H:i:s'),
        'ip' => $_SERVER['REMOTE_ADDR'],
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
        'request_uri' => $_SERVER['REQUEST_URI'],
        'method' => $_SERVER['REQUEST_METHOD'],
        'error' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ];

    error_log("Admin Panel Error: ".json_encode($errorDetails));

    if ($config['debug']) {
        echo json_encode([
            'error' => 'Internal Server Error',
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTrace()
        ], JSON_PRETTY_PRINT);
    } else {
        echo json_encode([
            'error' => 'Internal Server Error',
            'message' => 'An unexpected error occurred'
        ]);
    }
}
