# Kotchasan Framework Installation

## System Requirements

### Minimum Requirements

- **PHP**: 8.0 or higher
- **Composer**: 2.0 or higher
- **Web Server**: Apache 2.4+ or Nginx 1.18+
- **Database**: MySQL 5.7+ or MariaDB 10.3+
- **Memory**: 512MB RAM (recommended 1GB+)
- **Storage**: 100MB free space

### Recommended Requirements

- **PHP**: 8.1 or 8.2
- **Memory**: 2GB RAM
- **Storage**: 1GB free space
- **SSL Certificate**: For production

## Installation

### 1. Install via Composer (Recommended)

```bash
# Create new project
composer create-project kotchasan/framework my-project

# Navigate to project directory
cd my-project
```

### 2. Install via Git Clone

```bash
# Clone repository
git clone https://github.com/kotchasan/framework.git my-project
cd my-project

# Install dependencies
composer install
```

### 3. Install via Download

```bash
# Download ZIP file
wget https://github.com/kotchasan/framework/archive/main.zip
unzip main.zip
mv framework-main my-project
cd my-project

# Install dependencies
composer install
```

## Environment Configuration

### 1. Copy Environment File

```bash
cp .env.example .env
```

### 2. Edit .env File

```env
# Application
APP_NAME="Kotchasan Framework"
APP_ENV=local
APP_DEBUG=true
APP_URL=http://localhost

# Database
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=kotchasan
DB_USERNAME=root
DB_PASSWORD=

# Cache
CACHE_DRIVER=file
SESSION_DRIVER=file

# Mail
MAIL_MAILER=smtp
MAIL_HOST=localhost
MAIL_PORT=587
MAIL_USERNAME=null
MAIL_PASSWORD=null
MAIL_ENCRYPTION=null
```

### 3. Generate Application Key

```bash
php artisan key:generate
```

## Database Setup

### 1. Create Database

```sql
-- MySQL
CREATE DATABASE kotchasan CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- Or MariaDB
CREATE DATABASE kotchasan CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

### 2. Run Migrations

```bash
# Run all migrations
php artisan migrate

# Or run step by step
php artisan migrate --step
```

### 3. Seed Initial Data

```bash
# Seed initial data
php artisan db:seed

# Or seed specific seeder
php artisan db:seed --class=UserSeeder
```

## Web Server Configuration

### Apache Configuration

Create `.htaccess` file in `public` directory:

```apache
<IfModule mod_rewrite.c>
    RewriteEngine On

    # Handle Angular and other client-side routes
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^(.*)$ index.php [QSA,L]
</IfModule>

# Security headers
<IfModule mod_headers.c>
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
</IfModule>
```

### Nginx Configuration

```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /path/to/kotchasan/public;
    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.0-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.(?!well-known).* {
        deny all;
    }
}
```

## SSL Configuration (HTTPS)

### 1. Using Let's Encrypt (Recommended)

```bash
# Install Certbot
sudo apt install certbot python3-certbot-nginx

# Create SSL Certificate
sudo certbot --nginx -d your-domain.com
```

### 2. Using Self-Signed Certificate

```bash
# Generate private key
openssl genrsa -out server.key 2048

# Generate certificate
openssl req -new -x509 -key server.key -out server.crt -days 365
```

## Cache Configuration

### 1. File Cache (Default)

```env
CACHE_DRIVER=file
```

### 2. Redis Cache

```env
CACHE_DRIVER=redis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379
```

### 3. Memcached

```env
CACHE_DRIVER=memcached
MEMCACHED_HOST=127.0.0.1
MEMCACHED_PORT=11211
```

## Queue Configuration

### 1. Database Queue

```env
QUEUE_CONNECTION=database
```

```bash
# Create queue table
php artisan queue:table
php artisan migrate
```

### 2. Redis Queue

```env
QUEUE_CONNECTION=redis
```

## Mail Configuration

### 1. SMTP Configuration

```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=your-email@gmail.com
MAIL_PASSWORD=your-app-password
MAIL_ENCRYPTION=tls
```

### 2. Mailgun

```env
MAIL_MAILER=mailgun
MAILGUN_DOMAIN=your-domain.mailgun.org
MAILGUN_SECRET=your-mailgun-secret
```

## Logging Configuration

### 1. File Logging

```env
LOG_CHANNEL=stack
LOG_LEVEL=debug
```

### 2. Syslog

```env
LOG_CHANNEL=syslog
```

### 3. Custom Log Channel

```php
// config/logging.php
'channels' => [
    'custom' => [
        'driver' => 'single',
        'path' => storage_path('logs/custom.log'),
        'level' => 'debug',
    ],
],
```

## Installation Testing

### 1. Check PHP Version

```bash
php -v
# Should show PHP 8.0 or higher
```

### 2. Check Composer

```bash
composer --version
# Should show Composer 2.0 or higher
```

### 3. Check Required Extensions

```bash
php -m | grep -E "(pdo|mbstring|openssl|curl|json|zip)"
```

### 4. Test Application

```bash
# Start development server
php artisan serve

# Open browser to http://localhost:8000
```

## Troubleshooting

### Common Issues

#### 1. Permission Issues

```bash
# Set permissions
sudo chown -R www-data:www-data storage bootstrap/cache
sudo chmod -R 775 storage bootstrap/cache
```

#### 2. Composer Memory Limit

```bash
# Increase memory limit
php -d memory_limit=2G /usr/local/bin/composer install
```

#### 3. Database Connection Error

```bash
# Check database connection
php artisan tinker
>>> DB::connection()->getPdo();
```

#### 4. Cache Issues

```bash
# Clear cache
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

## Updates

### 1. Update Dependencies

```bash
composer update
```

### 2. Update Database Schema

```bash
php artisan migrate
```

### 3. Update Cache

```bash
php artisan optimize
```

## Backup

### 1. Database Backup

```bash
# MySQL
mysqldump -u root -p kotchasan > backup.sql

# MariaDB
mysqldump -u root -p kotchasan > backup.sql
```

### 2. File Backup

```bash
# Backup important files
tar -czf backup.tar.gz storage/ .env config/
```

## Production Installation

### 1. Environment Setup

```env
APP_ENV=production
APP_DEBUG=false
APP_URL=https://your-domain.com
```

### 2. Optimize Application

```bash
php artisan optimize
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### 3. Process Manager Setup

```bash
# Install Supervisor
sudo apt install supervisor

# Create configuration file
sudo nano /etc/supervisor/conf.d/kotchasan.conf
```

```ini
[program:kotchasan]
command=php /path/to/kotchasan/artisan queue:work
directory=/path/to/kotchasan
autostart=true
autorestart=true
user=www-data
redirect_stderr=true
stdout_logfile=/path/to/kotchasan/storage/logs/worker.log
```

---

**Note**: This documentation covers basic installation. For more information, please refer to the [Getting Started Guide](getting-started.md)

