/**
 * CMMS Manufacturing Platform - Main JavaScript
 * Handles mock data, interactions, and basic functionality
 */

// Mock Data Structure
const mockData = {
    dashboard: {
        systemStatus: 'online',
        lastSync: '30 seconds ago',
        uptime: 94.2,
        efficiency: 87,
        activeAlerts: 3
    },
    kpis: {
        equipmentUptime: {
            value: 94.2,
            description: 'Overall equipment effectiveness',
            change: 2.1,
            performance: 'Excellent',
            performanceClass: 'excellent'
        },
        activeWorkOrders: {
            value: 23,
            description: 'Currently in progress',
            change: -5,
            performance: 'Good',
            performanceClass: 'good'
        },
        overdueTasks: {
            value: 7,
            description: 'Require immediate attention',
            change: 2,
            performance: 'Needs Attention',
            performanceClass: 'warning'
        },
        criticalAlerts: {
            value: 3,
            description: 'High priority issues',
            change: -1,
            performance: 'Critical',
            performanceClass: 'critical'
        }
    },
    equipment: [
        {
            id: 1,
            name: 'CNC Machine #1',
            status: 'operational',
            image: 'CNC Machine',
            uptime: 98.5
        },
        {
            id: 2,
            name: 'Hydraulic Press #2',
            status: 'maintenance',
            image: 'Hydraulic Press',
            uptime: 85.2
        },
        {
            id: 3,
            name: 'Conveyor System #1',
            status: 'operational',
            image: 'Conveyor System',
            uptime: 96.8
        },
        {
            id: 4,
            name: 'Welding Robot #3',
            status: 'operational',
            image: 'Welding Robot',
            uptime: 94.1
        },
        {
            id: 5,
            name: 'Assembly Line #1',
            status: 'offline',
            image: 'Assembly Line',
            uptime: 0
        },
        {
            id: 6,
            name: 'Packaging Machine #2',
            status: 'maintenance',
            image: 'Packaging Machine',
            uptime: 0
        }
    ],
    alerts: [
        {
            id: 1,
            type: 'pressure',
            title: 'Hydraulic Press #2 Pressure Drop',
            description: 'Pressure has dropped below safe operating levels',
            equipment: 'Hydraulic Press #2',
            status: 'active',
            timestamp: '2 minutes ago',
            icon: 'exclamation-circle'
        },
        {
            id: 2,
            type: 'maintenance',
            title: 'Scheduled Maintenance Due',
            description: 'CNC Machine #1 maintenance is due in 2 days',
            equipment: 'CNC Machine #1',
            status: 'active',
            timestamp: '15 minutes ago',
            icon: 'calendar-check'
        },
        {
            id: 3,
            type: 'update',
            title: 'Software Update Available',
            description: 'New firmware version available for Welding Robot #3',
            equipment: 'Welding Robot #3',
            status: 'active',
            timestamp: '1 hour ago',
            icon: 'download'
        }
    ]
};

// Initialize the dashboard when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeDashboard();
    setupEventListeners();
});

/**
 * Initialize Dashboard - Load mock data and render UI
 */
function initializeDashboard() {
    console.log('Initializing CMMS Dashboard...');
    
    // Update header information
    updateHeaderInfo();
    
    // Update KPI cards
    updateKPICards();
    
    // Update equipment status
    updateEquipmentStatus();
    
    // Update alerts
    updateAlerts();
    
    console.log('Dashboard initialized successfully');
}

/**
 * Update Header Information
 */
function updateHeaderInfo() {
    // Update system status
    const statusDot = document.querySelector('.status-dot');
    if (statusDot && mockData.dashboard.systemStatus === 'online') {
        statusDot.classList.add('online');
    }
    
    // Update sync info
    const syncInfo = document.querySelector('.sync-info span:last-child');
    if (syncInfo) {
        syncInfo.textContent = `Last sync: ${mockData.dashboard.lastSync}`;
    }
    
    // Update uptime badge
    const uptimeBadge = document.querySelector('.uptime-badge span:last-child');
    if (uptimeBadge) {
        uptimeBadge.textContent = `${mockData.dashboard.uptime}% Uptime`;
    }
}

/**
 * Update KPI Cards with Mock Data
 */
function updateKPICards() {
    const kpiCards = document.querySelectorAll('.kpi-card');
    
    kpiCards.forEach((card, index) => {
        const kpiKey = Object.keys(mockData.kpis)[index];
        const kpiData = mockData.kpis[kpiKey];
        
        if (kpiData) {
            // Update value
            const valueElement = card.querySelector('.kpi-value');
            if (valueElement) {
                valueElement.textContent = kpiData.value;
            }
            
            // Update description
            const descElement = card.querySelector('.kpi-description');
            if (descElement) {
                descElement.textContent = kpiData.description;
            }
            
            // Update comparison
            const changeElement = card.querySelector('.change');
            if (changeElement) {
                changeElement.textContent = (kpiData.change >= 0 ? '+' : '') + kpiData.change + '%';
                changeElement.classList.add(kpiData.change >= 0 ? 'positive' : 'negative');
            }
            
            // Update performance status
            const statusElement = card.querySelector('.performance-status');
            if (statusElement) {
                statusElement.textContent = kpiData.performance;
                statusElement.className = `performance-status ${kpiData.performanceClass}`;
            }
        }
    });
}

/**
 * Update Equipment Status Section
 */
function updateEquipmentStatus() {
    // Count equipment by status
    const statusCounts = {
        total: mockData.equipment.length,
        operational: mockData.equipment.filter(e => e.status === 'operational').length,
        maintenance: mockData.equipment.filter(e => e.status === 'maintenance').length,
        offline: mockData.equipment.filter(e => e.status === 'offline').length
    };
    
    // Update status boxes
    const statusBoxes = document.querySelectorAll('.status-box');
    const statusLabels = ['Total', 'Operational', 'Maintenance', 'Offline'];
    const statusKeys = ['total', 'operational', 'maintenance', 'offline'];
    
    statusBoxes.forEach((box, index) => {
        const number = box.querySelector('.status-number');
        if (number) {
            number.textContent = statusCounts[statusKeys[index]];
        }
    });
    
    // Update equipment cards (show only first 3)
    const equipmentCardsContainer = document.querySelector('.equipment-cards');
    if (equipmentCardsContainer) {
        // Clear existing cards (except the first 3 which are in HTML)
        const existingCards = equipmentCardsContainer.querySelectorAll('.equipment-card');
        existingCards.forEach((card, index) => {
            if (index < 3 && mockData.equipment[index]) {
                updateEquipmentCard(card, mockData.equipment[index]);
            }
        });
    }
}

/**
 * Update Individual Equipment Card
 */
function updateEquipmentCard(cardElement, equipmentData) {
    // Update status badge
    const badge = cardElement.querySelector('.equipment-status-badge');
    if (badge) {
        const statusText = equipmentData.status.charAt(0).toUpperCase() + equipmentData.status.slice(1);
        badge.textContent = statusText;
        badge.className = `equipment-status-badge ${equipmentData.status === 'maintenance' ? 'maintenance' : ''}`;
    }
    
    // Update equipment name
    const nameElement = cardElement.querySelector('.equipment-info h3');
    if (nameElement) {
        nameElement.textContent = equipmentData.name;
    }
}

/**
 * Update System Alerts
 */
function updateAlerts() {
    const alertsList = document.querySelector('.alerts-list');
    if (!alertsList) return;
    
    // Clear existing alerts
    alertsList.innerHTML = '';
    
    // Add alerts from mock data
    mockData.alerts.forEach(alert => {
        const alertElement = createAlertElement(alert);
        alertsList.appendChild(alertElement);
    });
}

/**
 * Create Alert Element
 */
function createAlertElement(alertData) {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert-item ${alertData.status}`;
    
    const iconClass = alertData.icon || 'exclamation-circle';
    
    alertDiv.innerHTML = `
        <div class="alert-icon">
            <i class="fas fa-${iconClass}"></i>
        </div>
        <div class="alert-content">
            <h4>${alertData.title}</h4>
            <p>${alertData.description}</p>
            <span class="equipment-ref">${alertData.equipment}</span>
        </div>
        <div class="alert-status">
            <span class="status-badge ${alertData.status}">Active</span>
            <span class="alert-time">${alertData.timestamp}</span>
        </div>
    `;
    
    return alertDiv;
}

/**
 * Setup Event Listeners
 */
function setupEventListeners() {
    // Navigation menu items
    const navItems = document.querySelectorAll('.nav-item');
    navItems.forEach(item => {
        item.addEventListener('click', function() {
            // Remove active class from all items
            navItems.forEach(i => i.classList.remove('active'));
            // Add active class to clicked item
            this.classList.add('active');
        });
    });
    
    // Action buttons
    const actionButtons = document.querySelectorAll('.action-buttons .btn');
    actionButtons.forEach(button => {
        button.addEventListener('click', function() {
            const buttonText = this.textContent.trim();
            console.log(`Button clicked: ${buttonText}`);
            // Add your action handling here
            showNotification(`${buttonText} action triggered`);
        });
    });
    
    // Filter button
    const filterButton = document.querySelector('.filter-button');
    if (filterButton) {
        filterButton.addEventListener('click', function() {
            console.log('Filter clicked');
            showNotification('Filter panel would open here');
        });
    }
    
    // Search functionality
    const searchBox = document.querySelector('.search-box input');
    if (searchBox) {
        searchBox.addEventListener('input', function(e) {
            console.log('Search query:', e.target.value);
        });
    }
    
    // Equipment search
    const equipmentSearch = document.querySelector('.search-filter input');
    if (equipmentSearch) {
        equipmentSearch.addEventListener('input', function(e) {
            console.log('Equipment search:', e.target.value);
        });
    }
    
    // User profile dropdown
    const userProfile = document.querySelector('.user-profile');
    if (userProfile) {
        userProfile.addEventListener('click', function() {
            console.log('User profile clicked');
            showNotification('User menu would open here');
        });
    }
    
    // Sidebar user
    const sidebarUser = document.querySelector('.sidebar-user');
    if (sidebarUser) {
        sidebarUser.addEventListener('click', function() {
            console.log('Sidebar user clicked');
            showNotification('User settings would open here');
        });
    }
    
    // Settings icon
    const settingsIcon = document.querySelector('.settings');
    if (settingsIcon) {
        settingsIcon.addEventListener('click', function() {
            console.log('Settings clicked');
            showNotification('Settings panel would open here');
        });
    }
    
    // Notifications icon
    const notificationsIcon = document.querySelector('.notifications');
    if (notificationsIcon) {
        notificationsIcon.addEventListener('click', function() {
            console.log('Notifications clicked');
            showNotification('Notifications panel would open here');
        });
    }
}

/**
 * Show Notification (Simple notification system)
 */
function showNotification(message) {
    // Create notification element
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        bottom: 20px;
        right: 20px;
        background-color: #2563eb;
        color: white;
        padding: 16px 24px;
        border-radius: 8px;
        box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
        font-size: 14px;
        z-index: 1000;
        animation: slideIn 0.3s ease-out;
    `;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    // Remove notification after 3 seconds
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease-out';
        setTimeout(() => {
            notification.remove();
        }, 300);
    }, 3000);
}

// Add animations for notifications
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from {
            transform: translateX(400px);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    @keyframes slideOut {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(400px);
            opacity: 0;
        }
    }
`;
document.head.appendChild(style);

// Simulate real-time updates (for demo purposes)
function simulateRealtimeUpdates() {
    setInterval(() => {
        // Update last sync time
        const syncInfo = document.querySelector('.sync-info span:last-child');
        if (syncInfo) {
            syncInfo.textContent = 'Last sync: just now';
        }
        
        // Simulate occasional data changes
        if (Math.random() > 0.8) {
            console.log('Simulating data update...');
        }
    }, 30000); // Update every 30 seconds
}

// Start real-time simulation
simulateRealtimeUpdates();

console.log('CMMS Dashboard JavaScript loaded successfully');
