export class NotificationSystem {
    constructor(config) {
        this.config = config;
        this.channels = new Map();
        this.initializeChannels();
    }

    async initializeChannels() {
        // Discord
        if (this.config.discord.webhookUrl) {
            this.channels.set('discord', {
                send: async (message) => {
                    const response = await fetch(this.config.discord.webhookUrl, {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({
                            embeds: [{
                                title: 'New Order',
                                description: message,
                                color: 0x00ff00
                            }]
                        })
                    });
                    return response.ok;
                }
            });
        }

        // Telegram
        if (this.config.telegram.token && this.config.telegram.chatId) {
            this.channels.set('telegram', {
                send: async (message) => {
                    const response = await fetch(
                        `https://api.telegram.org/bot${this.config.telegram.token}/sendMessage`,
                        {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                chat_id: this.config.telegram.chatId,
                                text: message,
                                parse_mode: 'HTML'
                            })
                        }
                    );
                    return response.ok;
                }
            });
        }

        // LINE Official Account
        if (this.config.line.channelAccessToken) {
            this.channels.set('line', {
                send: async (message) => {
                    const response = await fetch('https://api.line.me/v2/bot/message/broadcast', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'Authorization': `Bearer ${this.config.line.channelAccessToken}`
                        },
                        body: JSON.stringify({
                            messages: [{
                                type: 'text',
                                text: message
                            }]
                        })
                    });
                    return response.ok;
                }
            });
        }
    }

    async notifyNewOrder(order) {
        const message = this.formatOrderMessage(order);
        const notifications = [];

        for (const [name, channel] of this.channels) {
            notifications.push(
                channel.send(message)
                    .catch(error => {
                        console.error(`${name} notification failed:`, error);
                        return false;
                    })
            );
        }

        return Promise.all(notifications);
    }

    formatOrderMessage(order) {
        return `
🆕 New Order #${order.id}
👤 Customer: ${order.customerName}
📱 Phone: ${order.phone}
📍 Address: ${order.address}

🛍️ Items:
${order.items.map(item => `- ${item.name} x${item.quantity} (${item.price}฿)`).join('\n')}

💰 Total: ${order.total}฿
💳 Payment: ${order.paymentMethod}
🚚 Delivery Notes: ${order.deliveryNotes || 'None'}
        `.trim();
    }
}
